/*
 * Decompiled with CFR 0.152.
 */
package org.hsqldb.persist;

import java.io.File;
import java.io.IOException;
import org.hsqldb.Database;
import org.hsqldb.HsqlException;
import org.hsqldb.NumberSequence;
import org.hsqldb.Session;
import org.hsqldb.Table;
import org.hsqldb.error.Error;
import org.hsqldb.lib.FileAccess;
import org.hsqldb.lib.FileArchiver;
import org.hsqldb.lib.HashMap;
import org.hsqldb.lib.Iterator;
import org.hsqldb.persist.Crypto;
import org.hsqldb.persist.DataFileCache;
import org.hsqldb.persist.DataFileDefrag;
import org.hsqldb.persist.HsqlDatabaseProperties;
import org.hsqldb.persist.RAShadowFile;
import org.hsqldb.persist.ScriptRunner;
import org.hsqldb.persist.TextCache;
import org.hsqldb.scriptio.ScriptReaderBase;
import org.hsqldb.scriptio.ScriptReaderDecode;
import org.hsqldb.scriptio.ScriptReaderText;
import org.hsqldb.scriptio.ScriptWriterBase;
import org.hsqldb.scriptio.ScriptWriterEncode;
import org.hsqldb.scriptio.ScriptWriterText;

public class Log {
    private HsqlDatabaseProperties properties;
    private String fileName;
    private Database database;
    private FileAccess fa;
    ScriptWriterBase dbLogWriter;
    private String scriptFileName;
    private String logFileName;
    private boolean filesReadOnly;
    private long maxLogSize;
    private int writeDelay;
    private int scriptFormat;
    private DataFileCache cache;
    private HashMap textCacheList = new HashMap();

    Log(Database db) {
        this.database = db;
        this.fa = db.logger.getFileAccess();
        this.fileName = db.getPath();
        this.properties = db.getProperties();
    }

    void initParams() {
        this.maxLogSize = (long)this.database.logger.propLogSize * 1024L * 1024L;
        this.scriptFormat = 0;
        this.writeDelay = this.database.logger.propWriteDelay;
        this.filesReadOnly = this.database.isFilesReadOnly();
        this.scriptFileName = this.fileName + ".script";
        this.logFileName = this.fileName + ".log";
    }

    void open() {
        this.initParams();
        int state = this.properties.getDBModified();
        switch (state) {
            case 1: {
                this.deleteNewAndOldFiles();
                this.processScript();
                this.processLog();
                this.close(false);
                if (this.cache != null) {
                    this.cache.open(this.filesReadOnly);
                }
                this.reopenAllTextCaches();
                break;
            }
            case 2: {
                this.renameNewDataFile();
                this.renameNewBackup();
                this.renameNewScript();
                this.deleteLog();
                this.properties.setDBModified(0);
            }
            case 0: {
                this.processScript();
                if (!this.isAnyCacheModified()) break;
                this.properties.setDBModified(1);
                this.close(false);
                if (this.cache != null) {
                    this.cache.open(this.filesReadOnly);
                }
                this.reopenAllTextCaches();
            }
        }
        this.openLog();
        if (!this.filesReadOnly) {
            this.properties.setDBModified(1);
        }
    }

    void close(boolean script) {
        this.closeLog();
        this.deleteNewAndOldFiles();
        this.writeScript(script);
        this.closeAllTextCaches(script);
        if (this.cache != null) {
            this.cache.close(true);
        }
        this.properties.setDBModified(2);
        this.deleteLog();
        if (script) {
            this.deleteBackup();
            this.deleteData();
        } else {
            try {
                this.backupData();
                this.renameNewBackup();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        this.renameNewScript();
        this.properties.setDBModified(0);
    }

    void shutdown() {
        this.synchLog();
        if (this.cache != null) {
            this.cache.close(false);
        }
        this.closeAllTextCaches(false);
        this.closeLog();
    }

    void deleteNewAndOldFiles() {
        this.fa.removeElement(this.fileName + ".data" + ".old");
        this.fa.removeElement(this.fileName + ".data" + ".new");
        this.fa.removeElement(this.fileName + ".backup" + ".new");
        this.fa.removeElement(this.scriptFileName + ".new");
    }

    void deleteBackup() {
        this.fa.removeElement(this.fileName + ".backup");
    }

    void deleteData() {
        this.fa.removeElement(this.fileName + ".data");
    }

    void backupData() throws IOException {
        if (this.database.logger.propIncrementBackup) {
            this.fa.removeElement(this.fileName + ".backup");
            return;
        }
        if (this.fa.isStreamElement(this.fileName + ".data")) {
            FileArchiver.archive(this.fileName + ".data", this.fileName + ".backup.new", this.database.logger.getFileAccess(), 1);
        }
    }

    void renameNewDataFile() {
        if (this.fa.isStreamElement(this.fileName + ".data.new")) {
            this.fa.renameElement(this.fileName + ".data.new", this.fileName + ".data");
        }
    }

    void renameNewBackup() {
        if (this.fa.isStreamElement(this.fileName + ".backup.new")) {
            this.fa.renameElement(this.fileName + ".backup.new", this.fileName + ".backup");
        }
    }

    void renameNewScript() {
        if (this.fa.isStreamElement(this.scriptFileName + ".new")) {
            this.fa.renameElement(this.scriptFileName + ".new", this.scriptFileName);
        }
    }

    void deleteNewScript() {
        this.fa.removeElement(this.scriptFileName + ".new");
    }

    void deleteNewBackup() {
        this.fa.removeElement(this.fileName + ".backup.new");
    }

    void deleteLog() {
        this.fa.removeElement(this.logFileName);
    }

    boolean isAnyCacheModified() {
        if (this.cache != null && this.cache.isFileModified()) {
            return true;
        }
        return this.isAnyTextCacheModified();
    }

    void checkpoint(boolean defrag) {
        if (this.filesReadOnly) {
            return;
        }
        if (this.cache == null) {
            defrag = false;
        } else if (this.forceDefrag()) {
            defrag = true;
        }
        if (defrag) {
            try {
                this.defrag();
                return;
            }
            catch (Exception e) {
                this.database.logger.checkpointDisabled = true;
                return;
            }
        }
        boolean result = this.checkpointClose();
        if (result) {
            this.checkpointReopen();
        }
    }

    boolean checkpointClose() {
        if (this.filesReadOnly) {
            return true;
        }
        this.deleteNewAndOldFiles();
        try {
            this.writeScript(false);
        }
        catch (HsqlException e) {
            this.deleteNewScript();
            return false;
        }
        try {
            if (this.cache != null) {
                this.cache.close(true);
                this.cache.backupFile();
            }
        }
        catch (Exception ee) {
            this.deleteNewScript();
            this.deleteNewBackup();
            try {
                if (!this.cache.isFileOpen()) {
                    this.cache.open(false);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            return false;
        }
        this.properties.setDBModified(2);
        this.closeLog();
        this.deleteLog();
        this.renameNewScript();
        this.renameNewBackup();
        try {
            this.properties.setDBModified(0);
        }
        catch (Exception exception) {
            // empty catch block
        }
        return true;
    }

    boolean checkpointReopen() {
        if (this.filesReadOnly) {
            return true;
        }
        try {
            if (this.cache != null) {
                this.cache.open(false);
            }
            if (this.dbLogWriter != null) {
                this.openLog();
            }
            this.properties.setDBModified(1);
        }
        catch (Exception e) {
            return false;
        }
        return true;
    }

    public void defrag() {
        if (this.cache.fileFreePosition == 32L) {
            return;
        }
        this.database.logger.logInfoEvent("defrag start");
        try {
            DataFileDefrag dfd = this.cache.defrag();
        }
        catch (HsqlException e) {
            this.database.logger.logSevereEvent("defrag failure", e);
            throw e;
        }
        catch (Throwable e) {
            this.database.logger.logSevereEvent("defrag failure", e);
            throw Error.error(129, e);
        }
        this.database.logger.logInfoEvent("defrag end");
    }

    boolean forceDefrag() {
        long limit = (long)this.database.logger.propCacheDefragLimit * this.cache.getFileFreePos() / 100L;
        long lostSize = this.cache.freeBlocks.getLostBlocksSize();
        return limit > 0L && lostSize > limit;
    }

    boolean hasCache() {
        return this.cache != null;
    }

    DataFileCache getCache() {
        if (this.cache == null) {
            this.cache = new DataFileCache(this.database, this.fileName);
            this.cache.open(this.filesReadOnly);
        }
        return this.cache;
    }

    void setLogSize(int megas) {
        this.maxLogSize = (long)megas * 1024L * 1024L;
    }

    int getScriptType() {
        return this.scriptFormat;
    }

    int getWriteDelay() {
        return this.writeDelay;
    }

    void setWriteDelay(int delay) {
        this.writeDelay = delay;
        if (this.dbLogWriter != null) {
            this.synchLog();
            this.dbLogWriter.setWriteDelay(delay);
        }
    }

    public void setIncrementBackup(boolean val) {
        if (this.cache != null) {
            this.cache.setIncrementBackup(val);
            this.cache.fileModified = true;
        }
    }

    void writeStatement(Session session, String s) {
        try {
            this.dbLogWriter.writeLogStatement(session, s);
        }
        catch (IOException e) {
            throw Error.error(29, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.checkpointRequired = true;
        }
    }

    void writeInsertStatement(Session session, Table t, Object[] row) {
        try {
            this.dbLogWriter.writeInsertStatement(session, t, row);
        }
        catch (IOException e) {
            throw Error.error(29, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.checkpointRequired = true;
        }
    }

    void writeDeleteStatement(Session session, Table t, Object[] row) {
        try {
            this.dbLogWriter.writeDeleteStatement(session, t, row);
        }
        catch (IOException e) {
            throw Error.error(29, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.checkpointRequired = true;
        }
    }

    void writeSequenceStatement(Session session, NumberSequence s) {
        try {
            this.dbLogWriter.writeSequenceStatement(session, s);
        }
        catch (IOException e) {
            throw Error.error(29, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.checkpointRequired = true;
        }
    }

    void writeCommitStatement(Session session) {
        try {
            this.dbLogWriter.writeCommitStatement(session);
        }
        catch (IOException e) {
            throw Error.error(29, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.checkpointRequired = true;
        }
    }

    void synchLog() {
        if (this.dbLogWriter != null) {
            this.dbLogWriter.sync();
        }
    }

    void openLog() {
        if (this.filesReadOnly) {
            return;
        }
        Crypto crypto = this.database.logger.getCrypto();
        try {
            this.dbLogWriter = crypto == null ? new ScriptWriterText(this.database, this.logFileName, false, false, false) : new ScriptWriterEncode(this.database, this.logFileName, crypto);
            this.dbLogWriter.setWriteDelay(this.writeDelay);
            this.dbLogWriter.start();
        }
        catch (Exception e) {
            throw Error.error(29, this.logFileName);
        }
    }

    synchronized void closeLog() {
        if (this.dbLogWriter != null) {
            this.dbLogWriter.close();
        }
    }

    void writeScript(boolean full) {
        this.deleteNewScript();
        Crypto crypto = this.database.logger.getCrypto();
        ScriptWriterText scw = crypto == null ? new ScriptWriterText(this.database, this.scriptFileName + ".new", full, true, false) : new ScriptWriterEncode(this.database, this.scriptFileName + ".new", full, crypto);
        scw.writeAll();
        scw.close();
    }

    private void processScript() {
        ScriptReaderBase scr = null;
        try {
            if (this.fa.isStreamElement(this.scriptFileName)) {
                Crypto crypto = this.database.logger.getCrypto();
                scr = crypto == null ? new ScriptReaderText(this.database, this.scriptFileName) : new ScriptReaderDecode(this.database, this.scriptFileName, crypto);
                Session session = this.database.sessionManager.getSysSessionForScript(this.database);
                scr.readAll(session);
                scr.close();
            }
        }
        catch (Throwable e) {
            if (scr != null) {
                scr.close();
                if (this.cache != null) {
                    this.cache.close(false);
                }
                this.closeAllTextCaches(false);
            }
            this.database.logger.logWarningEvent("Script processing failure", e);
            if (e instanceof HsqlException) {
                throw (HsqlException)e;
            }
            if (e instanceof IOException) {
                throw Error.error(29, e);
            }
            if (e instanceof OutOfMemoryError) {
                throw Error.error(72);
            }
            throw Error.error(40, e);
        }
    }

    private void processLog() {
        if (this.fa.isStreamElement(this.logFileName)) {
            ScriptRunner.runScript(this.database, this.logFileName);
        }
    }

    private void restoreBackup() {
        if (this.database.logger.propIncrementBackup) {
            this.restoreBackupIncremental();
            return;
        }
        DataFileCache.deleteOrResetFreePos(this.database, this.fileName + ".data");
        try {
            FileArchiver.unarchive(this.fileName + ".backup", this.fileName + ".data", this.database.logger.getFileAccess(), 1);
        }
        catch (Throwable t) {
            throw Error.error(t, 29, 115, new Object[]{t.getMessage(), this.fileName + ".backup"});
        }
    }

    private void restoreBackupIncremental() {
        try {
            if (this.fa.isStreamElement(this.fileName + ".backup")) {
                RAShadowFile.restoreFile(this.fileName + ".backup", this.fileName + ".data");
                this.deleteBackup();
            }
        }
        catch (IOException e) {
            throw Error.error(29, this.fileName + ".backup");
        }
    }

    DataFileCache openTextCache(Table table, String source, boolean readOnlyData, boolean reversed) {
        this.closeTextCache(table);
        if (this.database.getType() != "res:" && !this.properties.isPropertyTrue("textdb.allow_full_path")) {
            if (source.indexOf("..") != -1) {
                throw Error.error(33, source);
            }
            String path = new File(new File(this.database.getPath() + ".properties").getAbsolutePath()).getParent();
            if (path != null) {
                source = path + File.separator + source;
            }
        }
        TextCache c = new TextCache(table, source);
        c.open(readOnlyData || this.filesReadOnly);
        this.textCacheList.put(table.getName(), c);
        return c;
    }

    void closeTextCache(Table table) {
        TextCache c = (TextCache)this.textCacheList.remove(table.getName());
        if (c != null) {
            try {
                c.close(true);
            }
            catch (HsqlException hsqlException) {
                // empty catch block
            }
        }
    }

    private void closeAllTextCaches(boolean script) {
        Iterator it = this.textCacheList.values().iterator();
        while (it.hasNext()) {
            TextCache textCache = (TextCache)it.next();
            if (script && !textCache.table.isDataReadOnly()) {
                textCache.purge();
                continue;
            }
            textCache.close(true);
        }
    }

    private void reopenAllTextCaches() {
        Iterator it = this.textCacheList.values().iterator();
        while (it.hasNext()) {
            ((TextCache)it.next()).reopen();
        }
    }

    private boolean isAnyTextCacheModified() {
        Iterator it = this.textCacheList.values().iterator();
        while (it.hasNext()) {
            if (!((TextCache)it.next()).isFileModified()) continue;
            return true;
        }
        return false;
    }
}

